/* -*-C-*-
 ##############################################################################
 #
 # File:        demo/zoom.c
 # RCS:         $Id: zoom.c,v 1.10 1998/07/20 22:59:31 ericb Exp $
 # Description: Host program for doing inline FFTs
 # Created:     March 20, 1996
 # Language:    C
 # Package:     E1432
 #
 # Copyright (C) 1996 - 1998, Hewlett-Packard Company, all rights reserved.
 #
 ##############################################################################
 #
 #
 # Revisions:
 #
 #
 #  This program demonstrates how to put the E1432 into zoom mode such that
 #  frequency data is calculated only in the band: center freq +/- span/2.
 #  The E1432 in zoom mode multiplies the input data by a complex sine wave
 #  whose frequency is determined by the center frequency set.  This digital
 #  "local oscillator" centers the spectrum on the center frequency.  The
 #  complex time output from the LO is displayed along with its FFT.
 #
 ##############################################################################
 */
#include <math.h>
#include <stdlib.h>		/* For exit */
#include <stdio.h>		/* For printf */
#include <unistd.h>		/* For sleep */
#include "e1432.h"
#include "xplot.h"
#include "err1432.h"

#define TRAILER 
/*#define FREQ_ONLY*/

#define WIDTH           140
#define HEIGHT          80 
#define NUM_CHANS	16

#define RANGE		1.0
#define BLOCK_SIZE	1024
#define SPAN		1280
#define CENT_FREQ	2000
#define CLK_FREQ	65536
#define AVG_MODE	E1432_AVG_NONE
#define AVG_NUMBER	10
#define AVG_UPDATE	1
#define AVG_WEIGHT	5.0

/* Wrap this around all the many function calls which might fail */
#define	DEBUG(s)	s
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	DEBUG((void) printf("Error: %s returned %d\n", #func, _s));\
	return _s;\
    }\
} while (0)
#endif

int
main(void)
{
    int  i, j, k, nchan;
    LONGSIZ32 count;
    SHORTSIZ16 la = 8;
    SHORTSIZ16 chan_list[NUM_CHANS];
    SHORTSIZ16 inputs;
    E1432ID hw;
    struct e1432_hwconfig cf;
    SHORTSIZ16 error;
    SHORTSIZ16 avgMode;
    FLOATSIZ32 data[2*BLOCK_SIZE];
    FLOATSIZ32 *real[NUM_CHANS];
    FLOATSIZ32 *imag[NUM_CHANS];
    FLOATSIZ32 *freq[NUM_CHANS], *freqBuf, sum;
    long points = BLOCK_SIZE;
    long freq_points, freq_blocksize;
    char *plotid[3 * NUM_CHANS];
    int row, col, id;  
    char geometry[80];
    char title[80];
    float temp, floatRange;
    struct e1432_trailer trailer;
    char *semabinloc = "/opt/e1432/lib/sema.bin";
    float sample_freq;
    double lo_freq, hi_freq, bin_delta;
    int lo_bin, hi_bin, mag2;
    float span, cent_freq, top_freq, clock_freq;
    LONGSIZ32 block_size;

    for(i=0; i < NUM_CHANS; i++)
    {
	real[i] = NULL;
	imag[i] = NULL;
	freq[i] = NULL;
	plotid[i] = NULL;
	plotid[i + NUM_CHANS] = NULL;
	plotid[i + 2 * NUM_CHANS] = NULL;
    }

    avgMode = AVG_MODE;

    if(avgMode == E1432_AVG_NONE || avgMode == E1432_AVG_LIN)  
	mag2 = 0;
    else
	mag2 = 1;

    if(mag2)
	freq_blocksize = BLOCK_SIZE;
    else
	freq_blocksize = 2 * BLOCK_SIZE;

    temp = (float)(points - 1);

    if(e1432_init_io_driver()) {
       (void) printf("e1432_init_io_driver() failed\n");
       exit(0);
    }

    CHECK(e1432_print_errors(0));

    /* install the downloadable code */
    (void) printf("Checking for E1432 with firmware at logical address %d ... ",
									la);
    (void) fflush(stdout);
    error = e1432_get_hwconfig(1, &la, &cf);
    CHECK(e1432_print_errors(1));
    if (error)
    {
	(void) printf("Not found.\n\n");
	(void) printf("Installing firmware from %s into E1432 at la %d ... ",
					   semabinloc, la);
	(void) fflush(stdout);
	error = e1432_install(1, &la, 0, semabinloc);
	if (error)
	{
	    (void) printf("\ne1432_install failed and returned error %s\n");
	    exit(0);
	}
	(void) printf("Done.\n");
    }
    else
    {
	(void) printf("Found.\n");
    }
    (void) printf("\n");

    CHECK(e1432_assign_channel_numbers(1, &la, &hw));

    /* Create channel group */
    error = e1432_get_hwconfig(1, &la, &cf);
    if(error) {
        (void) printf("error in e1432_get_hwconfig(): %d\n", error);
        exit(0);
    }
    nchan = cf.input_chans;

    (void) printf("Found %d input channels\n", nchan);

    /* Create channel group */
    for(i=0; i<nchan; i++)
       chan_list[i] = E1432_INPUT_CHAN(i+1);

    inputs = e1432_create_channel_group(hw, nchan, chan_list);
    if (inputs >= 0)
    {
	(void) printf("e1432_create_channel_group inputs returned %d\n",
			    inputs);
	return -1;
    }

#if 0
    CHECK(e1432_set_internal_debug(hw, inputs, 0x400)); 
#endif

    /* Initialize hardware and measurment things */
    CHECK(e1432_set_analog_input(hw, inputs,
				     E1432_INPUT_MODE_VOLT,
				     E1432_INPUT_HIGH_NORMAL,
				     E1432_ANTI_ALIAS_ANALOG_ON,
				     E1432_COUPLING_AC, 
				     RANGE));

    CHECK(e1432_get_range(hw, chan_list[0], &floatRange));
    CHECK(e1432_set_data_size(hw, inputs, E1432_DATA_SIZE_32));
    CHECK(e1432_set_avg_mode(hw, inputs, AVG_MODE));
    CHECK(e1432_set_avg_number(hw, inputs, AVG_NUMBER));
    CHECK(e1432_set_avg_update(hw, inputs, AVG_UPDATE));
    CHECK(e1432_set_avg_weight(hw, inputs, AVG_WEIGHT));
    CHECK(e1432_set_blocksize(hw, inputs, BLOCK_SIZE)); 
    CHECK(e1432_get_blocksize(hw, inputs, &block_size)); 
    CHECK(e1432_set_data_mode(hw, inputs, E1432_BLOCK_MODE));
    CHECK(e1432_set_window(hw, inputs, E1432_WINDOW_HANNING));
    CHECK(e1432_set_calc_data(hw, inputs, E1432_DATA_FREQ));  
    CHECK(e1432_set_trigger_channel(hw, chan_list[0], E1432_CHANNEL_ON)); 
    CHECK(e1432_set_trigger_mode(hw, inputs, E1432_TRIGGER_MODE_LEVEL));
    CHECK(e1432_set_auto_trigger(hw, inputs, E1432_MANUAL_TRIGGER));
    CHECK(e1432_set_trigger_slope(hw, inputs, E1432_TRIGGER_SLOPE_POS));
    CHECK(e1432_set_trigger_level(hw, inputs, E1432_TRIGGER_LEVEL_LOWER, 0.0));
    CHECK(e1432_set_trigger_level(hw, inputs, E1432_TRIGGER_LEVEL_UPPER, 0.0));

    /* ALWAYS SET ZOOM MODE BEFORE SETTING SPAN SINCE SPAN WILL BE ROUNDED
     * TO A LEGAL VALUE FOR THE MODE
     */
    CHECK(e1432_set_clock_freq(hw, inputs, CLK_FREQ));
    CHECK(e1432_get_clock_freq(hw, inputs, &clock_freq));
    CHECK(e1432_set_center_freq(hw, inputs, CENT_FREQ));
    CHECK(e1432_set_zoom(hw, inputs, E1432_ZOOM_ON));
    CHECK(e1432_set_span(hw, inputs, SPAN));
    CHECK(e1432_get_span(hw, inputs, &span));
    CHECK(e1432_get_center_freq(hw, inputs, &cent_freq));
    CHECK(e1432_get_clock_freq(hw, inputs, &sample_freq));
      
    /* calculate the number of points to display such that the combination
     * of center frequency and span will not display any negative frequencies
     * or frequencies outside the range: center frequency +/- span/2.
     */
     
    bin_delta = 1.28 * span / (double)block_size;  /* freq change per bin */

    /* calculate the lowest and highest bin numbers to use while maintaining
     * 80dB alias protection
     */
    freq_points = block_size / 1.28 + 1;
    lo_bin = block_size / 2 - span / (2 * bin_delta) ;
    
    hi_bin = lo_bin + freq_points - 1;

    lo_freq = cent_freq - span / 2;
    if(lo_freq < 0.0)
    {
	lo_bin -=  lo_freq / bin_delta;
	lo_freq = 0.0;
    }

    top_freq = clock_freq / 2.56;
    hi_freq = cent_freq + span / 2;
    if(hi_freq > top_freq)
    {
	hi_bin -= (hi_freq - top_freq) / bin_delta;
	hi_freq = top_freq;
    }

    freq_points = hi_bin - lo_bin + 1;

    printf("sample freq = %g, center freq = %g, span = %g, block_size = %d\n", 
	sample_freq, cent_freq, span, block_size);

    /* set up xplot displays */
    row = col = 0;
    id = 0;

    for(i=0; i < nchan; i++) {
      if(real[i] == NULL) {
        real[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) * points);
        if(!real[i]) {
          (void) printf("Can't malloc real array of %d points\n", points);
          exit(0);
        }
	for(j=0; j < points; j++) {
	   real[i][j] = 0.0;
	}
      }

      if(imag[i] == NULL) {
        imag[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) * points);
        if(!imag[i]) {
          (void) printf("Can't malloc imag array of %d points\n", points);
          exit(0);
        }
	for(j=0; j < points; j++) {
	   imag[i][j] = 0.0;
	}
      }

      if(freq[i] == NULL) {
        freq[i] = (FLOATSIZ32 *)malloc(sizeof(FLOATSIZ32) * 2 * freq_blocksize);
        if(!freq[i]) {
          (void) printf("Can't malloc freq array of %d points\n", freq_blocksize);
          exit(0);
        }
	for(j=0; j < 2 * freq_blocksize; j++) {
	   freq[i][j] = 0.0;
	}
      }

      if(plotid[id] == NULL) { 
	 if(mag2)
            freqBuf = &(freq[i][lo_bin]);
	 else
            freqBuf = &(freq[i][2*lo_bin]);

        (void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
				(WIDTH + 20) * col, (HEIGHT + 40) * row ); 
	(void) sprintf(title, "Real %d", i + 1);

        plotid[id] = xplot_init_plot(real[i], points, temp, floatRange, 
			-floatRange, GENERIC_TRACE, geometry, title);
			    
        (void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
				(WIDTH + 20) * col, (HEIGHT + 40) * (row + 1));
	(void) sprintf(title, "Imag %d", i + 1);

        plotid[id+8] = xplot_init_plot(imag[i], points, temp, floatRange, 
			-floatRange, GENERIC_TRACE, geometry, title);
			    
        (void) sprintf( geometry, "%dx%d+%d+%d", WIDTH, HEIGHT,
			(WIDTH + 20) * col, (HEIGHT + 40) * (row  + 2)); 
	(void) sprintf(title, "Freq %d", i + 1);

        plotid[id+16] = xplot_init_plot(freqBuf, freq_points, 
		hi_freq, 0.0, -100.0, FREQ_LOG_MAG_TRACE, geometry, title);
      }
      xplot_change_yautoscale(plotid[id], 0);
      xplot_change_yautoscale(plotid[id+8], 0);
      xplot_change_yautoscale(plotid[id+16], 0);
      xplot_set_xscale(plotid[id], (float)0.0, temp);
      xplot_set_xscale(plotid[id+8], (float)0.0, temp);
      xplot_set_xscale(plotid[id+16], lo_freq, hi_freq);
      xplot_set_yscale(plotid[id], floatRange, -floatRange);
      xplot_set_yscale(plotid[id+8], floatRange, -floatRange);
      xplot_set_yscale(plotid[id+16], 10.0, -100.0);
      xplot_change_xlabel(plotid[id], "Samples");
      xplot_change_xlabel(plotid[id+8], "Samples");
      xplot_change_xlabel(plotid[id+16], "Hz");
      xplot_change_ylabel(plotid[id], "Volts");
      xplot_change_ylabel(plotid[id+8], "Volts");
      xplot_change_ylabel(plotid[id+16], "dBV");
      xplot_repaint(plotid[id]); 
      xplot_repaint(plotid[id+8]); 
      xplot_repaint(plotid[id+16]); 
        
      col++;
      id++;
      if((col % 8) == 0) {
          col = 0;
          row +=3;
          id += 16;
      }
    }

#ifdef TRAILER
    CHECK(e1432_set_append_status(hw, inputs, E1432_APPEND_STATUS_ON));
#endif

#ifdef FREQ_ONLY
	CHECK(e1432_set_enable(hw, inputs, 
				E1432_ENABLE_TYPE_TIME, E1432_ENABLE_OFF));
#endif 

    /* Start measurement */
    CHECK(e1432_init_measure(hw, inputs));
    
    for (;;)
    {
	/* Wait for block available */
	while(e1432_block_available(hw, inputs) <= 0)
		; 

#ifndef FREQ_ONLY
	for(i=0; i < nchan; i++) {	/* read time data */
            error = e1432_read_float32_data(hw, chan_list[i],
				E1432_TIME_DATA, data, 2 * points, 
				&trailer, &count);
            if(error) {
   	      printf("ERROR: time read_float32 error = %d, count = %d\n",
						error, count);
            }

	    for(j=0; j < points ; j++)
	    {
		real[i][j] = data[2*j];
		imag[i][j] = data[2*j+1];
	    }
	}

#endif

	id = 0;
        for(i=0; i < nchan; i++) {	/* get frequency data */
	    error = e1432_read_float32_data(hw, chan_list[i], 
			E1432_FREQ_DATA, freq[i], freq_blocksize, 
			&trailer, &count);
            if(error || count != freq_blocksize) {
   	      (void) printf("ERROR: freq read_float32 error = %d, count = %d\n",
							error, count);
            }

	    if(mag2)
                freqBuf = &(freq[i][lo_bin]);
	    else
                freqBuf = &(freq[i][2*lo_bin]);
	    
	    /* make the 0 dB point full scale */
	    for(j=0; j < 2 * freq_points; j++)
		freqBuf[j] /= floatRange;

	    if(!mag2)  
	    {
		/* do a magnitude squared of all of the complex points */
		for(j=0,k=0; j < freq_points; j++)
		{
		    sum = freqBuf[k] * freqBuf[k];  /* real part */
		    k++;
		    sum += freqBuf[k] * freqBuf[k]; /* imag part */
		    k++;
		    freqBuf[j] = sum;
		}
	    }

	    /* convert to dBV */
	    for(j=0; j < freq_points; j++)
	    {
		if(freqBuf[j] < 1e-23)	/* limit really small values */
		{
		    freqBuf[j] = 10.0 * log10(1e-23);
		}
		else
		{
		    freqBuf[j] = 10.0 * log10(freqBuf[j]);
		}
	    }

#ifndef FREQ_ONLY
            xplot_check_events(plotid[id]);
            xplot_data_update(plotid[id]);
            xplot_check_events(plotid[id+8]);
            xplot_data_update(plotid[id+8]);
#endif
            xplot_check_events(plotid[id+16]);
            xplot_data_update(plotid[id+16]);    
	    if((++id%8)==0) id += 16;
        }
    }
    /*NOTREACHED*/
    return 0;
}

